<?php

namespace App\Http\Controllers;

use App\Models\Group;
use App\Models\GroupMember;
use App\Models\User;
use Illuminate\Http\Request;

class GroupController extends Controller
{
    // list groups
    public function index()
    {
        $groups = Group::withCount(['members as active_members_count' => function ($q) {
            $q->where('status','active');
        }])->orderBy('created_at', 'desc')->paginate(15);

        return view('admin.groups.index', compact('groups'));
    }

    // show create form
    public function create()
    {
        return view('admin.groups.create');
    }

    // store new group
    public function store(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:191|unique:groups,name',
            'notes' => 'nullable|string',
        ]);

        $group = Group::create($request->only('name','notes'));

        return redirect()->route('admin.groups.show', $group->id)
            ->with('success', 'Group created. Add members to the group.');
    }

    // show group details + members
    public function show(Group $group)
    {
        $group->load(['members.user']);
        // provide a list of eligible members to add (members role only and not already in this group)
        $eligible = User::where('role', 'member')
            ->whereDoesntHave('groupMembers', function($q) use ($group) {
                $q->where('group_id', $group->id);
            })->orderBy('name')->get();

        return view('admin.groups.show', compact('group','eligible'));
    }

    // edit group
    public function edit(Group $group)
    {
        return view('admin.groups.edit', compact('group'));
    }

    // update group
    public function update(Request $request, Group $group)
    {
        $request->validate([
            'name' => 'required|string|max:191|unique:groups,name,'.$group->id,
            'notes' => 'nullable|string',
            'status' => 'required|in:active,inactive',
        ]);

        $group->update($request->only('name','notes','status'));

        return redirect()->route('admin.groups.show', $group->id)
            ->with('success', 'Group updated.');
    }

    // delete group
    public function destroy(Group $group)
    {
        $group->delete();
        return redirect()->route('admin.groups.index')->with('success','Group removed.');
    }

    // add member to group
    public function addMember(Request $request, Group $group)
    {
        $request->validate([
            'user_id' => 'required|exists:users,id',
        ]);

        $user = User::findOrFail($request->user_id);

        // ensure user role is 'member'
        if ($user->role !== 'member') {
            return back()->with('error','Only users with role "member" can be added to groups.');
        }

        // check duplicate
        $exists = GroupMember::where('group_id', $group->id)
                    ->where('user_id', $user->id)->exists();
        if ($exists) {
            return back()->with('error','Member already in group.');
        }

        GroupMember::create([
            'group_id' => $group->id,
            'user_id' => $user->id,
            'status' => 'active'
        ]);

        return back()->with('success','Member added.');
    }

    // remove member
    public function removeMember(Group $group, GroupMember $member)
    {
        // ensure belongs to the group
        if ($member->group_id !== $group->id) {
            return back()->with('error','Member does not belong to this group.');
        }

        $member->delete();

        return back()->with('success','Member removed.');
    }

    // toggle member status (active/inactive)
    public function toggleMemberStatus(Group $group, GroupMember $member)
    {
        if ($member->group_id !== $group->id) {
            return back()->with('error','Member does not belong to this group.');
        }

        $member->status = $member->status === 'active' ? 'inactive' : 'active';
        $member->save();

        return back()->with('success','Member status updated.');
    }
}
