<?php

namespace App\Http\Controllers;

use App\Models\Loan;
use App\Models\IndividualLoanRepayment;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class LoanRepaymentController extends Controller
{
    // =============================
    // MEMBER submits repayment
    // =============================
    public function memberStore(Request $request, Loan $loan)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'notes' => 'nullable|string'
        ]);

     IndividualLoanRepayment::create([
    'loan_id'   => $loan->id,
    'member_id' => Auth::id(),
    'amount'    => $request->amount,
    'status'    => 'pending',
    'notes'     => $request->notes
]);


        return back()->with('success', 'Repayment request submitted. Waiting for approval.');
    }

    // =============================
    // ADMIN records repayment directly
    // =============================
    public function adminStore(Request $request, Loan $loan)
    {
        $request->validate([
            'amount' => 'required|numeric|min:1',
            'notes' => 'nullable|string'
        ]);

    $repayment = IndividualLoanRepayment::create([
    'loan_id'   => $loan->id,
    'member_id' => $loan->user_id,
    'admin_id'  => Auth::id(),
    'amount'    => $request->amount,
    'balance_before' => $loan->balance,
    'balance_after'  => $loan->balance - $request->amount,
    'status'    => 'approved',
    'notes'     => $request->notes
]);

        // Reduce loan balance
        $loan->balance -= $request->amount;
        if ($loan->balance <= 0) {
            $loan->balance = 0;
            $loan->status = 'closed';
        }
        $loan->save();

        return redirect()->route('admin.loans.show', $loan->id)
            ->with('success', 'Repayment recorded successfully.');
    }

    // =============================
    // ADMIN approves member repayment
    // =============================
    public function approve(IndividualLoanRepayment $repayment)
    {
        $loan = $repayment->loan;

        $repayment->status = 'approved';
        $repayment->admin_id = Auth::id();
        $repayment->save();

        // Update loan balance
        $loan->balance -= $repayment->amount;
        if ($loan->balance <= 0) {
            $loan->balance = 0;
            $loan->status = 'closed';
        }
        $loan->save();

        return back()->with('success', 'Repayment approved.');
    }

    public function reject(IndividualLoanRepayment $repayment)
    {
        $repayment->status = 'rejected';
        $repayment->admin_id = Auth::id();
        $repayment->save();

        return back()->with('error', 'Repayment rejected.');
    }
}
