<?php

namespace App\Http\Controllers;

use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Storage;
use App\Models\ShareTransaction;

class UserController extends Controller
{
    // 🧭 List all users
    public function index(Request $request)
    {
        $query = User::query();

        // 🔍 Search filter
        if ($request->has('search') && !empty($request->search)) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                  ->orWhere('email', 'like', "%{$search}%")
                  ->orWhere('phone', 'like', "%{$search}%")
                  ->orWhere('member_id', 'like', "%{$search}%");
            });
        }

        // 🎭 Role filter
        if ($request->has('role') && !empty($request->role)) {
            $query->where('role', $request->role);
        }

        // 📄 Paginate results (10 per page)
        $users = $query->orderBy('created_at', 'desc')->paginate(10);

        return view('admin.users.index', compact('users'));
    }

    // 🧍‍♂️ Show create form
    public function create()
    {
        return view('admin.users.create');
    }

    // 💾 Store new user (admin-created)
    public function store(Request $request)
    {
       $request->validate([
    'name'        => 'required|string|max:255',
    'email'       => 'required|email|unique:users,email',
    'phone'       => 'nullable|string|max:20',
    'address'     => 'nullable|string|max:255',
    'national_id' => 'nullable|string|max:50',
    'role'        => 'required|string|in:admin,member,loan_officer,accountant',
    'photo'       => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
]);


        // Collect only safe fields (member_id auto-generation handled here)
        $data = $request->only('name', 'email', 'phone', 'address', 'national_id', 'role');

        // Generate a secure temp password for admin-created users (so DB never receives NULL)
        $tempPassword = Str::random(10); // readable random string
        $data['password'] = Hash::make($tempPassword);

        // Generate member id if not provided
        $data['member_id'] = 'MBR-' . strtoupper(Str::random(6));
        $data['status'] = 'active';

        // 📸 Handle Profile Photo
        if ($request->hasFile('photo')) {
            $filename = time() . '.' . $request->photo->extension();
            $request->photo->storeAs('public/photos', $filename);
            $data['photo'] = $filename;
        }

        // Create user
        $user = User::create($data);

        // Return to list with a flash message showing the temp password so admin can give it to the user
        return redirect()->route('users.index')
            ->with('success', 'User created successfully.')
            ->with('temp_password', $tempPassword)
            ->with('temp_user_email', $user->email);
    }

    // ✏️ Edit user form
    public function edit(User $user)
    {
        return view('admin.users.edit', compact('user'));
    }

    // 🔄 Update user
    public function update(Request $request, User $user)
    {
       $request->validate([
    'name'        => 'required|string|max:255',
    'phone'       => 'nullable|string|max:20',
    'address'     => 'nullable|string|max:255',
    'national_id' => 'nullable|string|max:50',
    'role'        => 'required|string|in:admin,member,loan_officer,accountant',
    'photo'       => 'nullable|image|mimes:jpg,jpeg,png|max:2048',
]);


        $data = $request->only('name', 'phone', 'address', 'national_id', 'role');

        // 📸 Handle photo update
        if ($request->hasFile('photo')) {
            $filename = time() . '.' . $request->photo->extension();
            $request->photo->storeAs('public/photos', $filename);

            // delete old photo
            if ($user->photo) {
                Storage::delete('public/photos/' . $user->photo);
            }

            $data['photo'] = $filename;
        }

        $user->update($data);

        return redirect()->route('users.index')->with('success', 'User updated successfully.');
    }

    // 🗑 Delete user
    public function destroy(User $user)
    {
        // delete user photo
        if ($user->photo) {
            Storage::delete('public/photos/' . $user->photo);
        }

        $user->delete();
        return redirect()->route('users.index')->with('success', 'User deleted successfully.');
    }

    // 🔄 Update status
    public function updateStatus(User $user, $status)
    {
        $validStatuses = ['active', 'pending', 'inactive'];

        if (!in_array($status, $validStatuses)) {
            return redirect()->back()->with('error', 'Invalid status.');
        }

        $user->status = $status;
        $user->save();

        return redirect()->back()->with('success', "User status updated to {$status}.");
    }

    // 👁 User detail page
   public function show(User $user)
   {
       // Get all this user's accounts
       $accounts = $user->accounts()->with('type')->get();

       // Get latest 6 transactions across all accounts
       $transactions = \App\Models\Transaction::whereIn('account_id', $accounts->pluck('id'))
                           ->orderBy('created_at', 'desc')
                           ->limit(6)
                           ->get();


  // ✅ Total Shares Counted
    $totalShares = ShareTransaction::where('user_id', $user->id)->sum('shares');

    // ✅ Total Amount Invested in Shares
    $totalShareValue = ShareTransaction::where('user_id', $user->id)->sum('amount');

       return view('admin.users.show', compact('user', 'accounts', 'transactions', 'totalShares', 'totalShareValue'));
   }


   public function resetPassword(User $user)
{
    // Generate new temporary password
    $newPassword = Str::random(10);

    // Update the user
    $user->password = Hash::make($newPassword);
    $user->save();

    // Redirect back with password preview
    return redirect()->back()->with([
        'success' => 'Password reset successfully.',
        'new_password' => $newPassword,
    ]);
}

}
