<?php

namespace App\Services;

use App\Models\LoanType;
use App\Models\User;
use App\Models\Group;
use App\Models\Account;
use App\Models\Transaction;
use Carbon\Carbon;

class LoanEligibilityService
{
    /**
     * Check rules for a user or group.
     * $subject can be User or Group model instance.
     * Returns array: [ 'ok' => bool, 'errors' => [] ]
     */
    public function check(LoanType $loanType, $subject): array
    {
        $rules = $loanType->rules ?? [];
        $errors = [];

        // Example supported rule keys:
        // min_savings_percent (e.g., 50)
        // min_daily_saving_days (e.g., 90)
        // require_group_of (e.g., 5)
        // forbid_if_active_loan (bool)

        // 1) If loan type is group loan, subject must be Group
        if ($loanType->is_group_loan && !$subject instanceof Group) {
            $errors[] = 'This loan type requires a group.';
            return ['ok' => false, 'errors' => $errors];
        }

        // 2) If loan type is individual & subject is Group, error
        if (!$loanType->is_group_loan && $subject instanceof Group) {
            $errors[] = 'This loan type is for individuals only.';
            return ['ok' => false, 'errors' => $errors];
        }

        // 3) Check forbid_if_active_loan
        if (!empty($rules['forbid_if_active_loan'])) {
            if ($subject instanceof \App\Models\User) {
                $active = $subject->loans()->whereIn('status', ['approved','active','pending'])->exists();
                if ($active) $errors[] = 'Member has an active or pending loan.';
            } else { // Group
                $active = \App\Models\Loan::where('group_id', $subject->id)
                    ->whereIn('status', ['approved','active','pending'])->exists();
                if ($active) $errors[] = 'Group has an active or pending loan.';
            }
        }

        // 4) min_savings_percent -> subject must have savings accounts sum >= X% * principal (we don't know principal yet)
        // We'll support a check when principal provided (controller will call separate method with principal)
        // 5) min_daily_saving_days -> check transaction history
        if (!empty($rules['min_daily_saving_days'])) {
            $days = intval($rules['min_daily_saving_days']);
            if ($subject instanceof \App\Models\User) {
                // count distinct days with deposits in last N days
                $from = Carbon::now()->subDays($days);
                $countDays = Transaction::where('user_id', $subject->id)
                    ->where('type','deposit')
                    ->where('created_at', '>=', $from)
                    ->selectRaw('date(created_at) as day')
                    ->groupBy('day')
                    ->get()
                    ->count();

                if ($countDays < $days) {
                    $errors[] = "Member must have at least {$days} days of deposit history.";
                }
            } else {
                // Group-level check: ensure each member meets some baseline? For now skip or check group aggregate
                // We'll require groups to satisfy min members instead (handled below).
            }
        }

        // 6) require_group_of
        if (!empty($rules['require_group_of']) && $subject instanceof Group) {
            $min = intval($rules['require_group_of']);
            if ($subject->activeMembersCount() < $min) {
                $errors[] = "Group requires at least {$min} active members.";
            }
        }

        // 7) min_total_group_savings (optional)
        if (!empty($rules['min_total_group_savings']) && $subject instanceof Group) {
            // sum group members balances (assuming Account model has balance and user_id)
            $total = 0;
            foreach ($subject->members as $gm) {
                $user = $gm->user;
                $total += $user->accounts()->sum('balance');
            }
            if ($total < floatval($rules['min_total_group_savings'])) {
                $errors[] = "Group total savings must be at least " . number_format($rules['min_total_group_savings'],2);
            }
        }

        return ['ok' => count($errors) === 0, 'errors' => $errors];
    }

    /**
     * Validate principal against min_savings_percent rule when principal provided.
     * Returns array same format.
     */
    public function checkPrincipalSavings(LoanType $loanType, $subject, float $principal): array
    {
        $rules = $loanType->rules ?? [];
        $errors = [];

        if (!empty($rules['min_savings_percent'])) {
            $percent = floatval($rules['min_savings_percent']);

            if ($subject instanceof \App\Models\User) {
                $savings = $subject->accounts()->sum('balance');
                $required = ($percent / 100) * $principal;
                if ($savings < $required) {
                    $errors[] = "Member must have at least {$percent}% of the principal saved (UGX " . number_format($required,2) . ").";
                }
            } else { // group
                $savings = 0;
                foreach ($subject->members as $gm) {
                    $savings += $gm->user->accounts()->sum('balance');
                }
                $required = ($percent / 100) * $principal;
                if ($savings < $required) {
                    $errors[] = "Group must have at least {$percent}% of the principal saved (UGX " . number_format($required,2) . ").";
                }
            }
        }

        return ['ok' => count($errors) === 0, 'errors' => $errors];
    }
}
